import React, { useState } from 'react';
import {
  Page,
  Card,
  Collapsible,
  Button,
  Text,
  TextField,
  Layout
} from '@shopify/polaris';

function FAQs() {
  const [searchValue, setSearchValue] = useState('');
  const [openItems, setOpenItems] = useState({});

  const faqs = [
    {
      id: '1',
      question: 'How do I connect my Amazon account?',
      answer: 'Go to Amazon Account Settings, click "Authenticate Amazon Account" to link your Amazon seller account. You\'ll be redirected to Amazon\'s authorization page to grant permissions.'
    },
    {
      id: '2',
      question: 'How many products can I import with the FREE plan?',
      answer: 'The FREE plan allows you to import up to 25 Amazon SKUs to your Shopify store. This plan is valid for 30 days and includes basic import functionality.'
    },
    {
      id: '3',
      question: 'What\'s the difference between Inventory Sync and Price Sync?',
      answer: 'Inventory Sync automatically updates your Shopify product quantities based on your Amazon inventory levels. Price Sync keeps your Shopify prices in sync with your Amazon prices. Both features are available in paid plans only.'
    },
    {
      id: '4',
      question: 'How often does the app sync inventory and prices?',
      answer: 'For paid plans with sync features enabled, inventory and prices are updated every 24 hours automatically.'
    },
    {
      id: '5',
      question: 'Can I import products from multiple Amazon marketplaces?',
      answer: 'Currently, the app supports one Amazon marketplace per connection. If you sell on multiple Amazon sites (US, UK, DE, etc.), you\'ll need to switch between accounts or contact support at shopifyapps@infoshore.biz for enterprise solutions.'
    },
    {
      id: '6',
      question: 'What happens when my subscription expires?',
      answer: 'When your subscription expires, you\'ll no longer be able to import new products or use sync features. Existing imported products will remain in your Shopify store, but won\'t receive updates from Amazon.'
    },
    {
      id: '7',
      question: 'Can I customize how products are imported?',
      answer: 'Yes! In Settings, you can configure whether products should be published immediately, set fulfillment services, choose inventory policies, and enable/disable sync features.'
    },
    {
      id: '8',
      question: 'Is there a limit on product variants?',
      answer: 'Yes, there\'s a limit on product variants as in Shopify, a product can have a maximum of 100 variants. Each Amazon ASIN with multiple variations (size, color, etc.) will be imported as a single Shopify product with multiple variants.'
    },
    {
      id: '9',
      question: 'What if I need to import more than 5,000 SKUs?',
      answer: 'We support importing more than 5,000 Amazon SKUs. Please contact our team via email for a custom quote with all the details of your requirements.'
    },
    {
      id: '10',
      question: 'Can I uninstall the app after importing products?',
      answer: 'Yes, if you only need one-time import without ongoing sync, you can uninstall the app after importing your products. The imported products will remain in your Shopify store.'
    }
  ];

  const filteredFaqs = faqs.filter(faq => 
    faq.question.toLowerCase().includes(searchValue.toLowerCase()) ||
    faq.answer.toLowerCase().includes(searchValue.toLowerCase())
  );

  const toggleItem = (id) => {
    setOpenItems(prev => ({
      ...prev,
      [id]: !prev[id]
    }));
  };

  return (
    <Page title="Frequently Asked Questions" subtitle="Find answers about importing Amazon products to Shopify">
      <Layout>
        <Layout.Section>
          <Card>
            <TextField
              label="Search FAQs"
              value={searchValue}
              onChange={setSearchValue}
              placeholder="Search for questions about Amazon import, sync, pricing..."
              clearButton
              onClearButtonClick={() => setSearchValue('')}
            />
          </Card>
        </Layout.Section>
        
        <Layout.Section>
          {filteredFaqs.map((faq) => (
            <Card key={faq.id}>
              <Button
                plain
                fullWidth
                textAlign="left"
                onClick={() => toggleItem(faq.id)}
              >
                <Text variant="headingMd" as="h3">{faq.question}</Text>
              </Button>
              <Collapsible
                open={openItems[faq.id]}
                id={`faq-${faq.id}`}
                transition={{ duration: '200ms', timingFunction: 'ease-in-out' }}
              >
                <div style={{ padding: '16px 0' }}>
                  <Text as="p">{faq.answer}</Text>
                </div>
              </Collapsible>
            </Card>
          ))}
        </Layout.Section>
        
        <Layout.Section>
          <Card>
            <Text variant="headingMd" as="h2">Still have questions?</Text>
            <Text as="p">Can't find what you're looking for? Our support team is here to help with your Amazon to Shopify import needs.</Text>
            <Button primary url="mailto:shopifyapps@infoshore.biz">
              Contact Support
            </Button>
          </Card>
        </Layout.Section>
      </Layout>
    </Page>
  );
}

export default FAQs;