import React, { useEffect, useState } from 'react';
import {
  Page,
  Layout,
  Card,
  Text,
  Button,
  Banner,
  Toast,
  Select,
  InlineStack,
  Box,
  Badge
} from '@shopify/polaris';

import { authUtils } from '../utils/auth';

const MARKETPLACE_OPTIONS = [
  { label: 'amazon.com', value: 'ATVPDKIKX0DER' },
  { label: 'amazon.ca', value: 'A2EUQ1WTGCTBG2' },
  { label: 'amazon.com.mx', value: 'A1AM78C64UM0Y8' },
  { label: 'amazon.co.uk', value: 'A1F83G8C2ARO7' },
  { label: 'amazon.in', value: 'A21TJRUUN4KGV' },
  { label: 'amazon.de', value: 'A1PA6795UKMFR9' },
  { label: 'amazon.es', value: 'A1RKKUPIHCS9HS' },
  { label: 'amazon.fr', value: 'A13V1IB3VIYZZH' },
  { label: 'amazon.it', value: 'APJ6JRA9NG5V4' },
  { label: 'amazon.eg', value: 'ARBP9OOSHTCHU' },
  { label: 'amazon.ae', value: 'A2VIGQ35RCS4UG' }
];

function AmazonAccount() {
  const [amzConfig, setAmzConfig] = useState(null);
  const [loading, setLoading] = useState(true);
  const DEFAULT_MARKETPLACE = MARKETPLACE_OPTIONS[0].value;
  const [marketplaceId, setMarketplaceId] = useState(DEFAULT_MARKETPLACE);
  const [toastActive, setToastActive] = useState(false);
  const [toastContent, setToastContent] = useState('');
  const [toastTone, setToastTone] = useState('success');
  const userData = authUtils.getUserData();
  const [subscriptionExpired, setSubscriptionExpired] = useState(userData?.expired === 1);

  useEffect(() => {
    const fetchConfig = async () => {
      setLoading(true);
      try {
        const token = authUtils.getToken();
        const response = await fetch(`${import.meta.env.VITE_API_BASE_URL}/amzconfig`, {
          headers: { Authorization: `Bearer ${token}` }
        });
        const data = await response.json();
        setAmzConfig(Array.isArray(data) && data.length === 0 ? null : data);
      } catch (e) {
        setAmzConfig(null);
      } finally {
        setLoading(false);
      }
    };
    fetchConfig();
  }, []);
  
  const handleAmazonAuth = async () => {
    const userData = authUtils.getUserData();
    if (!marketplaceId || !userData?.id) return;
    window.open(`https://shopify.infoshore.biz/xpert/authorize.php?key=${userData.id}&marketplace_id=${marketplaceId}`, '_self');
  };
  
  const handleInitiateFetch = async () => {
    try {
      const token = authUtils.getToken();
      const response = await fetch(`${import.meta.env.VITE_API_BASE_URL}/amzconfig/fetch`, {
        method: 'POST',
        headers: { Authorization: `Bearer ${token}` }
      });
      if (response.ok) {
        setToastContent('Amazon fetch initiated successfully!');
        setToastTone('success');
      } else {
        setToastContent('Failed to initiate Amazon fetch.');
        setToastTone('critical');
      }
    } catch {
      setToastContent('Error initiating fetch.');
      setToastTone('critical');
    } finally {
      setToastActive(true);
    }
  };
  
  return (
    <Page title="Amazon Account Settings" subtitle="Configure your Amazon integration">
      <Layout>
        {subscriptionExpired && (
          <Layout.Section>
            <Banner status="warning">
              <InlineStack vertical gap="200">
                <Text variant="headingMd">⚠️ Renewal Notice!</Text>
                <Text>
                  Your one-month subscription has expired. To continue importing Amazon SKUs and syncing inventory, please renew.
                </Text>
                <InlineStack gap="400">
                  <Button onClick={handleRenewSubscription}>Renew Now</Button>
                  <Button variant="secondary" onClick={() => window.location.href = '/plan-pricing'}>
                    Check Plan & Pricing
                  </Button>
                </InlineStack>
              </InlineStack>
            </Banner>
          </Layout.Section>
        )}

        <Layout.Section>
          <Card>
            <div style={{ padding: '20px' }}>
              {loading ? (
                <Text>Loading...</Text>
              ) : amzConfig ? (
                <>
                  <Text variant="headingMd" as="h2">Amazon Account Connected</Text>

                  <Box paddingBlock="200">
                    <Text as="span" fontWeight="medium">Marketplace ID:</Text>
                    <div style={{ marginTop: '6px' }}>
                      <Badge>{amzConfig?.marketplace_id || 'N/A'}</Badge>
                    </div>
                  </Box>

                  <Box paddingBlock="200">
                    <Text as="span" fontWeight="medium">Last Data Fetched:</Text>
                    <div style={{ marginTop: '6px' }}>
                      <Badge tone="info">{amzConfig?.lastdatafetched || 'N/A'}</Badge>
                    </div>
                  </Box>

                  <div style={{ marginTop: '20px' }}>
                    <Button primary onClick={handleInitiateFetch}>🔄 Initiate Amazon Fetch</Button>
                  </div>
                </>
              ) : (
                <>
                  <Text variant="headingMd" as="h2">Connect your Amazon Account</Text>
                  <div style={{ margin: '20px 0' }}>
                    <Select
                      label="Select Marketplace"
                      options={MARKETPLACE_OPTIONS}
                      value={marketplaceId}
                      onChange={setMarketplaceId}
                    />
                  </div>
                  <Button primary onClick={handleAmazonAuth} disabled={!marketplaceId}>
                    Add & Authenticate Amazon Account
                  </Button>
                </>
              )}
            </div>
          </Card>
        </Layout.Section>
      </Layout>

      {toastActive && (
        <Toast content={toastContent} onDismiss={() => setToastActive(false)} tone={toastTone} />
      )}
    </Page>
  );

}

export default AmazonAccount;