import React, { useState, useEffect } from 'react';
import {
  Page,
  Layout,
  Card,
  Text,
  Badge,
  Button,
  BlockStack,
  InlineStack,
  SkeletonBodyText,
  CalloutCard,
  Banner
} from '@shopify/polaris';
import axios from 'axios';
import { authUtils } from '../utils/auth';
import { useNavigate } from 'react-router-dom';
import amazonLogo from '../amazon.png';
import importImg from '../import.png';
import integrateImg from '../integearteapps.png';
import settingImg from '../setting.png';

function Dashboard() {
  const navigate = useNavigate();
  const [dashboardData, setDashboardData] = useState({
    products: 0,
    importInProgressProducts: 0,
    importedProducts: 0,
    planType: '',
    accountStatus: '',
    pricing: '',
    importCredit: 0,
    importConsumed: 0,
    amazonToken: ''
  });
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    fetchDashboardData();
  }, []);

  const fetchDashboardData = async () => {
    try {
      setLoading(true);
      setError(null);
      
      const baseURL = import.meta.env.VITE_API_BASE_URL;
      const apiKey = import.meta.env.VITE_API_KEY;
      // const urlParams = new URLSearchParams(window.location.search);
      // const apiKey = urlParams.get('tempcode');
      
      const response = await axios.post(`${baseURL}/authenticate`, {}, {
        params: {
          key: apiKey
        }
      });

      const { token, user } = response.data;
      
      // Store token and user data in localStorage
      authUtils.setAuthData(token, user);
      
      // Map API response to dashboard data according to specifications
      const mappedData = {
        products: user.productsCount || 0,
        importInProgressProducts: user.importInProgressProducts || 0,
        importedProducts: user.skuconsumed || 0,
        planType: user.membershiptype || 'free',
        accountStatus: user.membershiptype === 'paid' ? 'Premium' : 'Free',
        pricing: user.membershiptype === 'paid' ? '$19.99' : 'Free',
        importCredit: parseInt(user.skulimit) || 0,
        importConsumed: parseInt(user.skuconsumed) || 0,
        amazonToken: user.amzkey === 1 ? 'Valid' : 'Expired'
      };

      setDashboardData(mappedData);
      setLoading(false);
    } catch (error) {
      console.error('Error fetching dashboard data:', error);
      setError('Failed to load dashboard data. Please try again.');
      setLoading(false);
    }
  };

  return (
    
    <Page>
      <BlockStack gap="500">
        {error && (
          <Banner tone="critical">
            {error}
          </Banner>
        )}

        {/* Book Now Banner */}
        <CalloutCard
          title="Need a hand getting started?"
          illustration="https://apps1.infoshore.biz/updateaac/carts.png"
          primaryAction={{
            content: 'Book Now',
            onAction: () => window.open('https://calendly.com/madhu-infoshore/30min', '_blank')
          }}
        >
          <p>Book your free onboarding call and import your first product with ease!</p>
        </CalloutCard>

        {/* Conditional Import Banner */}
        {dashboardData.importedProducts === 0 && (
          <Banner tone="info">
            <Text as="span" variant="bodyMd">
              <span
                style={{ color: '#2c6ecb', cursor: 'pointer', textDecoration: 'underline' }}
                onClick={() => navigate('/products')}
              >
                Click here
              </span>{' '}
              to import your first products to Shopify!
            </Text>
          </Banner>
        )}

        <Layout>
          {/* Stats: Total, In Progress, Imported */}
          <Layout.Section>
            <Layout>
              {/* Shopify Products */}
              <Layout.Section variant="oneThird">
                <Card>
                  <BlockStack gap="100">
                    <Text as="h2" variant="headingMd" tone="success">Products</Text>
                    <InlineStack align="center">
                      <Text as="h2" variant="heading2xl">{dashboardData.products}</Text>
                    </InlineStack>
                  </BlockStack>
                </Card>
              </Layout.Section>

              {/* Import In Progress */}
              <Layout.Section variant="oneThird">
                <Card>
                  <BlockStack gap="100">
                    <Text as="h2" variant="headingMd" tone="caution">Import In Progress</Text>
                    <InlineStack align="center">
                      <Text as="h2" variant="heading2xl">{dashboardData.importInProgressProducts}</Text>
                    </InlineStack>
                  </BlockStack>
                </Card>
              </Layout.Section>

              {/* Imported Products */}
              <Layout.Section variant="oneThird">
                <Card>
                  <BlockStack gap="100">
                    <Text as="h2" variant="headingMd" tone="success">Imported Products</Text>
                    <InlineStack align="center">
                      <Text as="h2" variant="heading2xl">{dashboardData.importedProducts}</Text>
                    </InlineStack>
                  </BlockStack>
                </Card>
              </Layout.Section>
            </Layout>
          </Layout.Section>

          {/* Plan and Integration */}
          <Layout.Section>
            <Layout>
              {/* Plan */}
              <Layout.Section variant="twoThird">
                <Card>
                  <BlockStack gap="100">
                    <Badge tone={dashboardData.planType === 'free' ? 'attention' : 'success'}>
                      {dashboardData.planType === 'free' ? 'Free Plan' : 'Paid Plan'}
                    </Badge>
                    <Text variant="headingMd">Plan Details</Text>
                    <InlineStack align="space-between">
                      <Text>Pricing</Text>
                      <Text>{dashboardData.pricing}</Text>
                    </InlineStack>
                    <InlineStack align="space-between">
                      <Text>Import Credit</Text>
                      <Text>{dashboardData.importCredit}</Text>
                    </InlineStack>
                    <InlineStack align="space-between">
                      <Text>Import Consumed</Text>
                      <Text>{dashboardData.importConsumed}</Text>
                    </InlineStack>
                  </BlockStack>
                </Card>
              </Layout.Section>

              {/* Integration Card */}
              <Layout.Section variant="oneThird">
                <Card padding="none">
                  <a
                    href="https://apps.shopify.com/search?q=infoshore"
                    target="_blank"
                    rel="noopener noreferrer"
                    style={{ display: 'block' }}
                  >
                    <img
                      src={integrateImg}
                      alt="Integrate"
                      style={{
                        width: '100%',
                        height: '100%',
                        objectFit: 'cover',
                        display: 'block',
                        borderRadius: '8px'
                      }}
                    />
                  </a>
                </Card>
              </Layout.Section>
            </Layout>
          </Layout.Section>

          {/* Feature Cards: Import / Settings / Amazon */}
          <Layout.Section>
            <Layout>
              {/* Import Product */}
              <Layout.Section variant="oneThird">
                <Card>
                  <InlineStack align="center" gap="400">
                    {/* Left: Image */}
                    <img
                      src={importImg}
                      alt="Import"
                      style={{ width: 80, height: 80, objectFit: 'contain' }}
                    />

                    {/* Right: Text content */}
                    <BlockStack gap="200">
                      <Text variant="headingSm" fontWeight="bold">
                        Import Product
                      </Text>
                      <Text tone="subdued">
                        Preview and Import products into your Shopify store from Amazon.
                      </Text>
                      <Button size="slim" onClick={() => navigate('/products')}>Import Product</Button>
                    </BlockStack>
                  </InlineStack>
                </Card>
              </Layout.Section>

              {/* Settings */}
              <Layout.Section variant="oneThird">
                <Card>
                  <InlineStack align="center" gap="400">
                    <img
                      src={settingImg}
                      alt="Settings"
                      style={{ width: 80, height: 80, objectFit: 'contain', borderRadius: 8 }}
                    />
                    <BlockStack gap="200">
                      <Text variant="headingSm" fontWeight="bold">Settings</Text>
                      <Text tone="subdued">View and manage all the configurations of your account and store.</Text>
                      <Button size="slim" onClick={() => navigate('/settings')}>Manage Product</Button>
                    </BlockStack>
                  </InlineStack>
                </Card>
              </Layout.Section>

              {/* Amazon Account */}
              <Layout.Section variant="oneThird">
                <Card>
                  <InlineStack align="center" gap="400">
                    <img
                      src={amazonLogo}
                      alt="Amazon"
                      style={{ width: 80, height: 80, objectFit: 'contain', borderRadius: 8 }}
                    />
                    <BlockStack gap="200">
                      <Text variant="headingSm" fontWeight="bold">Amazon Account</Text>
                      <Text tone="subdued">View and manage all your Amazon Associate tags and marketplaces here.</Text>
                      <Button size="slim" onClick={() => navigate('/amazon-account')}>Amazon Account</Button>
                    </BlockStack>
                  </InlineStack>
                </Card>
              </Layout.Section>
            </Layout>
          </Layout.Section>
        </Layout>
      </BlockStack>
    </Page>
  );
}

export default Dashboard;