import React, { useState, useEffect } from 'react';
import {
  Page,
  Layout,
  Card,
  Text,
  Button,
  Badge,
  List,
  Grid,
  Banner,
  ButtonGroup
} from '@shopify/polaris';
import { authUtils } from '../utils/auth';

function PlanPricing() {
  const [currentPlan, setCurrentPlan] = useState('free');
  const [billingCycle, setBillingCycle] = useState('monthly');
  const [importLimit] = useState(25);
  const [importConsumed] = useState(0);
  const [userData, setUserData] = useState(null);
  const [showBanner, setShowBanner] = useState(false);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const data = authUtils.getUserData();
    setUserData(data);
    setLoading(false);
    setShowBanner(data?.membershiptype === 'free');
    if (data?.plan === 1 && data?.sync === 1) setCurrentPlan('bronze');
    else if (data?.plan === 2 && data?.sync === 1) setCurrentPlan('silver');
    else if (data?.plan === 3 && data?.sync === 1) setCurrentPlan('gold');
    else setCurrentPlan('free');
  }, []);

  // Helper to get tempcode/key
  const getTempCode = () => userData?.tempcode || userData?.id || '';
  const planDb = parseInt(userData?.plan || 0);

  // Plan type mapping
  const planTypeMap = { bronze: 1, silver: 2, gold: 3 };

  // Handle plan activation
  const handlePlanActivate = (planId) => {
    if (planId === 'free') return;
    const key = getTempCode();
    const plantype = planTypeMap[planId];
    let url = '';
    if (billingCycle === 'annual') {
      url = `https://shopify.infoshore.biz/xpert/upgradeannual.php?key=${key}&type=${plantype}`;
    } else {
      url = `https://shopify.infoshore.biz/xpert/upgrade2.php?key=${key}&type=${plantype}`;
    }
    window.location.href = url;
  };

  const planArray = [
    "free","bronze","silver","gold"
  ];

  const plans = {
    monthly: [
      {
        id: 'free',
        name: 'FREE',
        price: '$0',
        period: '',
        subtitle: '(25 Amazon SKUs)',
        validFor: 'Valid for 30 days',
        features: [
          '✓ Manage 25 SKUs',
          '✓ Import Products',
          '✗ Inventory Sync',
          '✗ Price Sync',
          '✗ Priority Customer Support'
        ],
        buttonText: 'Activated',
        buttonVariant: 'primary',
      },
      {
        id: 'bronze',
        name: 'BRONZE',
        price: '$20',
        period: '/mo',
        subtitle: '(500 Amazon SKUs)',
        billedAt: 'Billed at $20 per month',
        features: [
          '✓ Manage 500 SKUs',
          '✓ Import Products',
          '✓ Inventory Sync',
          '✓ Price Sync',
          '✗ Priority Customer Support'
        ],
        buttonText: 'Activate Now!',
        buttonVariant: 'secondary'
      },
      {
        id: 'silver',
        name: 'SILVER',
        price: '$30',
        period: '/mo',
        subtitle: '(2000 Amazon SKUs)',
        billedAt: 'Billed at $30 per month',
        features: [
          '✓ Manage 2000 SKUs',
          '✓ Import Products',
          '✓ Inventory Sync',
          '✓ Price Sync',
          '✓ Priority Customer Support'
        ],
        buttonText: 'Activate Now!',
        buttonVariant: 'secondary'
      },
      {
        id: 'gold',
        name: 'GOLD',
        price: '$60',
        period: '/mo',
        subtitle: '(5000 Amazon SKUs)',
        billedAt: 'Billed at $60 per month',
        features: [
          '✓ Manage 5000 SKUs',
          '✓ Import Products',
          '✓ Inventory Sync',
          '✓ Price Sync',
          '✓ Priority Customer Support'
        ],
        buttonText: 'Activate Now!',
        buttonVariant: 'secondary'
      }
    ],
    annual: [
      {
        id: 'free',
        name: 'FREE',
        price: '$0',
        period: '',
        subtitle: '(25 Amazon SKUs)',
        validFor: 'Valid for 30 days',
        features: [
          '✓ Manage 25 SKUs',
          '✓ Import Products',
          '✗ Inventory Sync',
          '✗ Price Sync',
          '✗ Priority Customer Support'
        ],
        buttonText: 'Activated',
        buttonVariant: 'primary',
      },
      {
        id: 'bronze',
        name: 'BRONZE',
        price: '$199',
        period: '/year',
        subtitle: '(500 Amazon SKUs)',
        billedAt: 'Billed at $199 per year (Save $40)',
        features: [
          '✓ Manage 500 SKUs',
          '✓ Import Products',
          '✓ Inventory Sync',
          '✓ Price Sync',
          '✗ Priority Customer Support'
        ],
        buttonText: 'Activate Now!',
        buttonVariant: 'secondary'
      },
      {
        id: 'silver',
        name: 'SILVER',
        price: '$299',
        period: '/year',
        subtitle: '(2000 Amazon SKUs)',
        billedAt: 'Billed at $299 per year (Save $60)',
        features: [
          '✓ Manage 2000 SKUs',
          '✓ Import Products',
          '✓ Inventory Sync',
          '✓ Price Sync',
          '✓ Priority Customer Support'
        ],
        buttonText: 'Activate Now!',
        buttonVariant: 'secondary'
      },
      {
        id: 'gold',
        name: 'GOLD',
        price: '$499',
        period: '/year',
        subtitle: '(5000 Amazon SKUs)',
        billedAt: 'Billed at $499 per year (Save $120)',
        features: [
          '✓ Manage 5000 SKUs',
          '✓ Import Products',
          '✓ Inventory Sync',
          '✓ Price Sync',
          '✓ Priority Customer Support'
        ],
        buttonText: 'Activate Now!',
        buttonVariant: 'secondary'
      }
    ]
  };

  const currentPlans = plans[billingCycle];

  const PlanCard = ({ plan }) => {
    const isActive = planArray[planDb] === plan.id;
    return (
      <Card>
        <div style={{ textAlign: 'center', padding: '24px', height: '100%' }}>
          <Text variant="headingLg" as="h3">{plan.name}</Text>
          <Text variant="bodySm" color="subdued">{plan.subtitle}</Text>
          <div style={{ margin: '20px 0' }}>
            <Text variant="heading2xl" as="p">{plan.price}</Text>
            {plan.period && <Text variant="bodySm" color="subdued">{plan.period}</Text>}
            {plan.validFor && <Text variant="bodySm" color="subdued">{plan.validFor}</Text>}
            {plan.billedAt && <Text variant="bodySm" color="subdued">{plan.billedAt}</Text>}
          </div>
          <div style={{ textAlign: 'left', marginBottom: '24px' }}>
            {plan.features.map((feature, index) => (
              <div key={index} style={{ marginBottom: '8px' }}>
                <Text variant="bodySm">{feature}</Text>
              </div>
            ))}
          </div>
          <Button {...(isActive ? { id: 'activate-button' } : {})}
            fullWidth
            primary={isActive}
            disabled={isActive}
            pressed={isActive}
            // tone={isActive ? 'success' : undefined}
            variant={isActive ? 'primary' : "secondary"}
            onClick={() => !isActive && handlePlanActivate(plan.id)}
          >
            {isActive ? 'Activated' : 'Activate Now!'}
          </Button>
        </div>
      </Card>
    );
  };

  if (loading) return null;

  return (
    <Page title="Monthly Plans With Inventory Sync Services">
      <Layout>
        {showBanner && (
          <Layout.Section>
            <Banner status="info">
              <Text>You have subscribed to the FREE plan. We support importing upto 25 SKUs under FREE plan. Choose any plan below to import upto 5,000 SKUs.</Text>
            </Banner>
          </Layout.Section>
        )}
        
        <Layout.Section>
          <div style={{ display: 'flex', justifyContent: 'center', marginBottom: '32px' }}>
            <ButtonGroup segmented>
              <Button
                tone={billingCycle !== 'monthly' ? 'success' : "secondary"}
                variant={billingCycle !== 'monthly' ? 'secondary' : "primary"}
                pressed={billingCycle === 'monthly'}
                onClick={() => setBillingCycle('monthly')}
              >
                Billed Monthly
              </Button>
              <Button
                variant={billingCycle !== 'annual' ? 'secondary' : "primary"}
                tone={billingCycle !== 'annual' ?  'success' : "secondary"}

                // tone={billingCycle === 'annual' ? 'success' : undefined}
                pressed={billingCycle === 'annual'}
                onClick={() => setBillingCycle('annual')}
              >
                Billed Annually
              </Button>
            </ButtonGroup>
          </div>
        </Layout.Section>
        <Layout.Section>
          <Grid>
            {currentPlans.map((plan) => (
              <Grid.Cell key={plan.id} columnSpan={{ xs: 6, sm: 3, md: 3, lg: 3, xl: 3 }}>
                <PlanCard plan={plan} />
              </Grid.Cell>
            ))}
          </Grid>
        </Layout.Section>
        
        <Layout.Section>
          <Card>
            <div style={{ padding: '20px', textAlign: 'center' }}>
              <Text variant="headingMd" as="h2">Need More Than 5,000 SKUs?</Text>
              <Text>Yes we do support more than 5,000 amazon SKUs. Drop us an email at <a href="mailto:shopifyapps@infoshore.biz">shopifyapps@infoshore.biz</a> for a quote with all the details.</Text>
              <div style={{ marginTop: '16px' }}>
                <Button url="mailto:shopifyapps@infoshore.biz">E-Mail Now!</Button>
              </div>
            </div>
          </Card>
        </Layout.Section>
        
        <Layout.Section>
          <Card>
            <div style={{ padding: '20px' }}>
              <div style={{ display: 'flex', justifyContent: 'space-between', marginBottom: '16px' }}>
                <Text>Import Limit: {importLimit}</Text>
                <Text>Import Consumed: {importConsumed}</Text>
              </div>
              <Text variant="bodySm" color="subdued">
                Please note that APP sync your shopify products with your Amazon seller account. If you just need to import one time you can just uninstall the app after imports.
              </Text>
            </div>
          </Card>
        </Layout.Section>
      </Layout>
    </Page>
  );
}

export default PlanPricing;