import React, { useState, useEffect, useCallback } from "react";
import wrap from "word-wrap";
import {
  Page,
  Card,
  Layout,
  IndexTable,
  Text,
  Badge,
  Button,
  Pagination,
  Select,
  Toast,
  useIndexResourceState,
  useSetIndexFiltersMode,
  IndexFilters,
  ButtonGroup,
  LegacyCard,
  SkeletonBodyText,
  Popover,
  Box,
  InlineStack,
  ActionList,
  EmptyState,
} from "@shopify/polaris";
import axios from "axios";
import { authUtils } from "../utils/auth";

function Products() {
  const [itemProducts, setItemProducts] = useState([
    `All`,
    `Ready To Import`,
    "Imported",
    "Import In Progress",
    "Error",
  ]);
  const [searchValue, setSearchValue] = useState("");
  const [selectedTab, setSelectedTab] = useState(0);
  const [products, setProducts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [perPage, setPerPage] = useState(20);
  const [totalProducts, setTotalProducts] = useState(0);
  const [firstProductNo, setFirstProductNo] = useState(0);
  const [lastProductNo, setLastProductNo] = useState(0);
  const [totalAllProducts, setTotalAllProducts] = useState(0);
  const [totalReadyToImport, setTotalReadyToImport] = useState(0);
  const [totalImported, setTotalImported] = useState(0);
  const [totalImportInProgress, setTotalImportInProgress] = useState(0);
  const [stats, setStats] = useState({
    all: 0,
    readyToImport: 0,
    imported: 0,
    importInProgress: 0,
  });
  const [actionPopoverActive, setActionPopoverActive] = useState(false);
  const { selectedResources, allResourcesSelected, handleSelectionChange } =
    useIndexResourceState(products);
  const [toastActive, setToastActive] = useState(false);
  const [toastContent, setToastContent] = useState("");
  const [toastTone, setToastTone] = useState("success");
  const { mode, setMode } = useSetIndexFiltersMode();
  const [productData ,setProductData]= useState(false);

  useEffect(() => {
    const status = getStatusFromTab();
    const isFiltered = searchValue.trim() !== "" || status !== "all";
    if (!isFiltered) {
      fetchProducts(currentPage, perPage);
    }
  }, []);

  const showToast = (content, tone = "success") => {
    setToastContent(content);
    setToastTone(tone);
    setToastActive(true);
  };

  const fetchProducts = async (page = 1, perPageValue = 20, query = "") => {
      setProductData(false);

    if (products.length === 0) {
      setLoading(true);
    }

    try {
      const baseURL = import.meta.env.VITE_API_BASE_URL;
      let apiUrl;
      let params = new URLSearchParams({
        per_page: perPageValue,
        page: page,
      });

      if (query && query.trim() !== "") {
        apiUrl = `${baseURL}/product/searchnew`;
        params.append("title", query.trim());
      } else {
        apiUrl = `${baseURL}/product`;
      }

      const response = await authUtils.authenticatedRequest(
        `${apiUrl}?${params.toString()}`,
        {
          method: "GET",
          headers: {
            "Content-Type": "application/json",
          },
        }
      );

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const data = await response.json();

      const formattedProducts = data.data.map((product) => ({
        id: product.product_id,
        title: wrap(product.title, { width: 40 }),
        status: product.status,
        block: Number(product.block) || 0,
        variants: product.variants?.length || 0,
        image: product.variants?.[0]?.imageurl || "/api/placeholder/40/40",
        sku: product.variants?.[0]?.sku || "",
        price: product.variants?.[0]?.price || "0.00",
        quantity: product.variants_count?.quantity || "0",
        variantCount: product.variants_count?.no_of_variants || "1",
      }));

      setProducts(formattedProducts);
      setTotalProducts(data.total);
      setTotalAllProducts(data.total);
      setTotalReadyToImport(data.ready_to_import || 0);
      setTotalImported(data.imported || 0);
      setTotalImportInProgress(data.import_in_progress || 0);
      setCurrentPage(data.current_page);
      setTotalPages(data.last_page);
      setFirstProductNo(data.from);
      setLastProductNo(data.to);
      setProductData(true);

      // Calculate stats based on actual data
      const imported = formattedProducts.filter(
        (p) => p.status === "Imported"
      ).length;
      const readyToImport = formattedProducts.filter(
        (p) => p.status === "Ready to Import"
      ).length;
      const importInProgress = formattedProducts.filter(
        (p) => p.status === "Import In Progress"
      ).length;

      setStats({
        all: data.total,
        readyToImport: readyToImport,
        imported: imported,
        importInProgress: importInProgress,
      });

      setLoading(false);
    } catch (error) {
      console.error("Error fetching products:", error);
      setLoading(false);
    }
  };

  const searchnew = async (
    page = 1,
    perPageValue = 20,
    query = "",
    status = ""
  ) => {
    try {
      setProductData(false);

      const baseURL = import.meta.env.VITE_API_BASE_URL;
      const apiUrl = `${baseURL}/product/searchnew`;

      const params = new URLSearchParams({
        per_page: perPageValue,
        page: page,
      });

      if (query && query.trim() !== "") {
        params.append("query", query.trim());
      }

      if (status && status !== "all") {
        params.append("status", status);
      }

      const response = await authUtils.authenticatedRequest(
        `${apiUrl}?${params}`,
        {
          method: "GET",
          headers: {
            "Content-Type": "application/json",
          },
        }
      );

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      setProductData(true);

      const data = await response.json();

      const formattedProducts = data.data.map((product) => ({
        id: product.product_id,
        title: wrap(product.title, { width: 40 }),
        status: product.status,
        block: Number(product.block) || 0,
        variants: product.variants?.length || 0,
        image: product.variants?.[0]?.imageurl || "/api/placeholder/40/40",
        sku: product.variants?.[0]?.sku || "",
        price: product.variants?.[0]?.price || "0.00",
        quantity: product.variants_count?.quantity || "0",
        variantCount: product.variants_count?.no_of_variants || "1",
      }));

      setProducts(formattedProducts);
      setTotalProducts(data.total);
      setCurrentPage(data.current_page);
      setTotalPages(data.last_page);
      setFirstProductNo(data.from);
      setLastProductNo(data.to);

      const imported = formattedProducts.filter(
        (p) => p.status === "Imported"
      ).length;
      const readyToImport = formattedProducts.filter(
        (p) => p.status === "Ready to Import"
      ).length;
      const importInProgress = formattedProducts.filter(
        (p) => p.status === "Import In Progress"
      ).length;

      setStats({
        all: data.total,
        readyToImport: readyToImport,
        imported: imported,
        importInProgress: importInProgress,
      });
    } catch (error) {
      console.error("Error searching products:", error);
    }
  };

  const handleBlockProduct = async (productIds) => {
    try {
      const baseURL = import.meta.env.VITE_API_BASE_URL;
      const apiUrl = `${baseURL}/product/block/${
        Array.isArray(productIds) ? productIds.join(",") : productIds
      }`;

      const response = await authUtils.authenticatedRequest(apiUrl, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();

      if (result.success) {
      } else {
        console.error("Block operation failed:", result.error);
      }
      const status = getStatusFromTab();
      searchnew(currentPage, perPage, searchValue, status);
      // searchnew();

    } catch (error) {
      console.error("Error blocking products:", error);
    }
  };

  const handleUnblockProduct = async (productIds) => {
    try {
      const baseURL = import.meta.env.VITE_API_BASE_URL;
      const apiUrl = `${baseURL}/product/unblock/${
        Array.isArray(productIds) ? productIds.join(",") : productIds
      }`;

      const response = await authUtils.authenticatedRequest(apiUrl, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const result = await response.json();

      if (result.success) {
        const status = getStatusFromTab();
        searchnew(currentPage, perPage, searchValue, status);
      } else {
        console.error("Unblock operation failed:", result.error);
      }
    } catch (error) {
      console.error("Error unblocking products:", error);
    }
  };

  const handleImportNow = async (productId) => {
    try {
      const baseURL = import.meta.env.VITE_API_BASE_URL;
      const apiUrl = `${baseURL}/product/push/${productId}`;

      const response = await authUtils.authenticatedRequest(apiUrl, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
      });

      const result = await response.json();

      if (!response.ok) {
        throw new Error(result?.msg || "Import failed");
      }

      showToast("Sent to Import process.", "success");

      searchnew(currentPage, perPage, searchValue, getStatusFromTab());
    } catch (error) {
      showToast(error.message || "Error importing product", "critical");
    }
  };

  const handleImportSelected = async () => {
    if (!selectedResources.length) return;

    try {
      const baseURL = import.meta.env.VITE_API_BASE_URL;
      const ids = selectedResources.join(",");
      const apiUrl = `${baseURL}/product/pushall`;

      const response = await authUtils.authenticatedRequest(apiUrl, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
      });

      const result = await response.json();

      if (!response.ok) {
        throw new Error(result?.msg || "Bulk import failed");
      }

      showToast("Selected products sent in Import process.", "success");

      searchnew(currentPage, perPage, searchValue, getStatusFromTab());
    } catch (error) {
      showToast(
        error.message || "Error importing selected products",
        "critical"
      );
    }
  };

  const handleDeleteSelected = async () => {
    if (!selectedResources.length) return;

    try {
      const baseURL = import.meta.env.VITE_API_BASE_URL;
      const ids = selectedResources.join(",");
      const apiUrl = `${baseURL}/product/remove/${ids}`;

      const response = await authUtils.authenticatedRequest(apiUrl, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
      });

      const result = await response.json();

      if (!response.ok) {
        throw new Error(result?.msg || "Delete failed");
      }

      setToast({
        open: true,
        content: `Deleted selected products successfully.`,
        tone: "success",
      });
    } catch (error) {
      setToast({
        open: true,
        content: error.message || "Error deleting selected products",
        tone: "critical",
      });
    }
  };

  const handleBlock = (productId) => {
    const product = products.find((p) => p.id === productId);
    if (product && product.block === 1) {
      handleUnblockProduct(productId);
    } else {
      handleBlockProduct(productId);
    }
  };

  const handleResetFilter = () => {
    setSearchValue("");
    setCurrentPage(1);
    fetchProducts(1, perPage, "");
  };

  const getStatusFromTab = (tabIndex = selectedTab) => {
    switch (tabIndex) {
      case 0:
        return "all";
      case 1:
        return "Ready to Import";
      case 2:
        return "Imported";
      case 3:
        return "Import In Progress";
      default:
        return "all";
    }
  };

  const handleTabChange = (tabIndex) => {
    setSelectedTab(tabIndex);
    setCurrentPage(1);
    const status = getStatusFromTab(tabIndex);
    searchnew(1, perPage, "", status);
  };

  const handleQueryChange = (tabIndex) => {
    setSearchValue(tabIndex);
    setCurrentPage(1);
    searchnew(1, perPage, tabIndex);
  };

  const handlePerPageChange = useCallback(
    (value) => {
      const newPerPage = parseInt(value);
      setPerPage(newPerPage);
      setCurrentPage(1);
      const status = getStatusFromTab();
      if (searchValue.trim() !== "" || status !== "all") {
        searchnew(1, newPerPage, searchValue, status);
      } else {
        fetchProducts(1, newPerPage, "");
      }
    },
    [searchValue, selectedTab]
  );

  const handlePreviousPage = () => {
    
    if (currentPage > 1) {
      const newPage = currentPage - 1;
      setCurrentPage(newPage);
      const status = getStatusFromTab();
      if (searchValue.trim() !== "" || status !== "all") {
        searchnew(newPage, perPage, searchValue, status);
      } else {
        fetchProducts(newPage, perPage, "");
      }
    }
  };

  const handleNextPage = () => {
    if (currentPage < totalPages) {
      const newPage = currentPage + 1;
      setCurrentPage(newPage);

      const status = getStatusFromTab();
      if (searchValue.trim() !== "" || status !== "all") {
        searchnew(newPage, perPage, searchValue, status);
      } else {
        fetchProducts(newPage, perPage, "");
      }
    }
  };

  const handleBlockSelected = () => {
    if (selectedResources.length > 0) {
      handleBlockProduct(selectedResources);
    }
  };

  const getBadgeTone = (status) => {
    switch (status.toLowerCase()) {
      case "ready to import":
        return "info";
      case "import in progress":
      case "imported":
        return "success";
      case "already exist":
        return "attention";
      case "error":
        return "critical";
      default:
        return "attention";
    }
  };

  const tabs = [
    { id: "all", content: `All ${totalAllProducts}`, panelID: "all-content" },
    {
      id: "ready",
      content: `Ready To Import ${totalReadyToImport}`,
      panelID: "ready-content",
    },
    {
      id: "imported",
      content: `Imported ${totalImported}`,
      panelID: "imported-content",
    },
    {
      id: "progress",
      content: `Import In Progress ${totalImportInProgress}`,
      panelID: "progress-content",
    },
  ];

  const resourceName = {
    singular: "product",
    plural: "products",
  };

  const rowMarkup = products.map(
    (
      { id, title, image, sku, variantCount, price, quantity, status, block },
      index
    ) => (
      <IndexTable.Row
        id={id}
        onClick={() => {console.log(id)}}
        key={id}
        selected={selectedResources.includes(id)}
        position={index}
      >
        <IndexTable.Cell>
          <div style={{ display: "flex", alignItems: "center", gap: "12px" }}>
            <div style={{ minWidth: 0, flex: 1 }}>
              <Text variant="bodySm" fontWeight="medium" breakWord>
                {title.split("\n").map((line, idx) => (
                  <span key={idx}>
                    {line}
                    <br />
                  </span>
                ))}
              </Text>
              <Text variant="bodySm" color="subdued">
                SKU: {sku}
              </Text>
            </div>
          </div>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd">{variantCount}</Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd">${price}</Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd">{quantity}</Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Badge tone={getBadgeTone(status)}>{status}</Badge>
        </IndexTable.Cell>
        <IndexTable.Cell>
          {status == 'Ready to Import'&&(
            <ButtonGroup>
              <Button
                size="slim"
                onClick={(e) => {
                  e.stopPropagation();
                  handleImportNow(id);
                }}
              >
                Import Now
              </Button>
              <Button
                size="slim"
                variant="secondary"
                onClick={(e) => {
                  e.stopPropagation();
                  handleBlock(id);
                }}
                tone={block === 1 ? "success" : "critical"}
              >
                {block === 1 ? "Unblock" : "Block"}
              </Button>
            </ButtonGroup>
          )} 
        </IndexTable.Cell>
      </IndexTable.Row>
    )
  );

  const actionMarkup = (
    <Popover
      active={actionPopoverActive}
      activator={
        <Button
          disclosure
          onClick={() => setActionPopoverActive(!actionPopoverActive)}
        >
          Actions
        </Button>
      }
      onClose={() => setActionPopoverActive(false)}
    >
      <ActionList
        items={[
          {
            content: "Import All Products",
            onAction: handleImportSelected,
          },
          { content: "Block Selected Products", onAction: handleBlockSelected },
          {
            content: "Unblock Selected Products",
            onAction: () => handleUnblockProduct(selectedResources),
          },
          {
            content: "Delete Selected Products",
            onAction: handleDeleteSelected,
          },
        ]}
      />
    </Popover>
  );

  let content ;
  if(!productData){
      content =    <Card>
                                       
                                       <LegacyCard subdued>
          <LegacyCard.Section>
            <SkeletonBodyText lines={2} />
          </LegacyCard.Section>
          <LegacyCard.Section>
            <SkeletonBodyText lines={2} />
          </LegacyCard.Section>
        </LegacyCard>
        <LegacyCard subdued>
          <LegacyCard.Section>
            <SkeletonBodyText lines={2} />
          </LegacyCard.Section>
          <LegacyCard.Section>
            <SkeletonBodyText lines={2} />
          </LegacyCard.Section>
        </LegacyCard>             
              <LegacyCard subdued>
          <LegacyCard.Section>
            <SkeletonBodyText lines={2} />
          </LegacyCard.Section>
          <LegacyCard.Section>
            <SkeletonBodyText lines={2} />
          </LegacyCard.Section>
        </LegacyCard>
            <LegacyCard subdued>
          <LegacyCard.Section>
            <SkeletonBodyText lines={2} />
          </LegacyCard.Section>
          <LegacyCard.Section>
            <SkeletonBodyText lines={2} />
          </LegacyCard.Section>
        </LegacyCard>
                             
                                         </Card>;
  }else if(products.length == 0){
    content =<Card>  <EmptyState title="No products found"
         
              image="https://cdn.shopify.com/s/files/1/0262/4071/2726/files/emptystate-files.png"
            >
              <p>No Products Found</p>
            </EmptyState>
            </Card> 
  }else{
    content =     <IndexTable
                resourceName={resourceName}
                itemCount={products.length}
                selectedItemsCount={
                  allResourcesSelected ? "All" : selectedResources.length
                }
                onSelectionChange={handleSelectionChange}
                headings={[
                  { title: "Title & SKU" },
                  { title: "# Variants" },
                  { title: "Price" },
                  { title: "Quantity" },
                  { title: "Status" },
                  { title: "Actions" },
                ]}
              >
                {rowMarkup}
              
              </IndexTable>
  }

  return (
    <Page title="Products" fullWidth>
      <Layout>
        <Layout.Section>
          {/* <Tabs tabs={tabs} selected={selectedTab} onSelect={handleTabChange}>
          </Tabs> */}
          <Box padding="400">
            <InlineStack align="space-between" wrap>
              {actionMarkup}
            </InlineStack>
            {/* Product Table */}
            <Box paddingBlockStart="400">
              
              <IndexFilters
                autoFocusSearchField={true}
                filteringAccessibilityTooltip="Search and filter (F)"
                queryValue={searchValue}
                onQueryChange={handleQueryChange}
                onQueryClear={() => {
                  handleResetFilter();
                }}
                cancelAction={{
                  type: "Cancel",
                  content: "cancel",
                }}
                tabs={tabs}
                selected={selectedTab}
                onSelect={handleTabChange}
                filters={[]}
                mode={mode}
                setMode={setMode}
                canCreateNewView={false}
              />
              {content}
          
            </Box>

            {/* Pagination */}
            <Box
            >
              <InlineStack align="space-between" wrap blockAlign="center">
                <Text variant="bodySm">
                  Showing {firstProductNo} to {lastProductNo} of {totalProducts}{" "}
                  entries
                </Text>

                <InlineStack gap="400" wrap blockAlign="center">
                  <InlineStack gap="200" blockAlign="center">
                    <Text variant="bodySm">Show</Text>
                    <Select
                      options={[
                        { label: "20", value: "20" },
                        { label: "50", value: "50" },
                        { label: "100", value: "100" },
                      ]}
                      value={perPage.toString()}
                      onChange={handlePerPageChange}
                    />
                    <Text variant="bodySm">entries</Text>
                  </InlineStack>
                  <Pagination
                    hasPrevious={currentPage > 1}
                    onPrevious={handlePreviousPage}
                    hasNext={currentPage < totalPages}
                    onNext={handleNextPage}
                    label={`Page ${currentPage} of ${totalPages}`}
                  />
                </InlineStack>
              </InlineStack>
            </Box>
          </Box>
        </Layout.Section>
      </Layout>

      {toastActive && (
        <Toast
          content={toastContent}
          onDismiss={() => setToastActive(false)}
          tone={toastTone}
        />
      )}
    </Page>
  );
}

export default Products;
