import React, { useState, useEffect } from 'react';
import {
  Page,
  Layout,
  Card,
  FormLayout,
  Button,
  Text,
  Select,
  Banner,
  Toast,
  Frame,
  BlockStack,
  InlineStack,
  SkeletonBodyText
} from '@shopify/polaris';
import { authUtils } from '../utils/auth';

function Settings() {
  const [publishStatus, setPublishStatus] = useState('Published');
  const [fulfillmentService, setFulfillmentService] = useState('Manual');
  const [inventoryPolicy, setInventoryPolicy] = useState('Shopify tracks this product\'s inventory');
  const [inventorySync, setInventorySync] = useState('Disable');
  const [priceSync, setPriceSync] = useState('Disable');
  const [autoImport, setAutoImport] = useState('Disable');
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState(null);
  const [toastActive, setToastActive] = useState(false);
  const [toastMessage, setToastMessage] = useState('');

  const publishOptions = [
    { label: 'Published', value: 'Published' },
    { label: 'Draft', value: 'Draft' }
  ];

  const fulfillmentOptions = [
    { label: 'Manual', value: 'manual' },
    { label: 'Amazon Marketplace Web', value: 'amazon_marketplace_web' }
  ];

  const inventoryPolicyOptions = [
    { label: 'Shopify tracks this product\'s inventory', value: 'shopify' },
    { label: 'Don\'t track inventory', value: 'continue' }
  ];

  const syncOptions = [
    { label: 'Enable', value: '1' },
    { label: 'Disable', value: '0' }
  ];

  useEffect(() => {
    fetchSettings();
  }, []);

  const fetchSettings = async () => {
    try {
      setLoading(true);
      setError(null);
      
      const baseURL = import.meta.env.VITE_API_BASE_URL;
      const response = await authUtils.authenticatedRequest(`${baseURL}/settings`);
      
      if (response.ok) {
        const data = await response.json();
        
        // Map API response to form state
        setPublishStatus(data.published === '1' ? 'Published' : 'Draft');
        setFulfillmentService(data.fulfilment_service || 'manual');
        setInventoryPolicy(data.inventory_policy || 'shopify');
        setInventorySync(data.inventory_sync || '0');
        setPriceSync(data.price_sync || '0');
        setAutoImport(data.autoimport || '0');
      } else {
        throw new Error('Failed to fetch settings');
      }
    } catch (error) {
      console.error('Error fetching settings:', error);
      setError('Failed to load settings. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  const handleSave = async () => {
    try {
      setSaving(true);
      setError(null);
      
      const baseURL = import.meta.env.VITE_API_BASE_URL;
      const requestData = {
        published: publishStatus === 'Published' ? '1' : '0',
        fulfilment_service: fulfillmentService,
        inventory_policy: inventoryPolicy,
        inventory_sync: inventorySync,
        price_sync: priceSync,
        autoimport: autoImport
      };
      
      const response = await authUtils.authenticatedRequest(`${baseURL}/settings`, {
        method: 'POST',
        body: JSON.stringify(requestData)
      });
      
      if (response.ok) {
        setToastMessage('Settings saved successfully!');
        setToastActive(true);
      } else {
        throw new Error('Failed to save settings');
      }
    } catch (error) {
      console.error('Error saving settings:', error);
      setError('Failed to save settings. Please try again.');
    } finally {
      setSaving(false);
    }
  };

  const toggleToastActive = () => setToastActive(!toastActive);

  const toastMarkup = toastActive ? (
    <Toast content={toastMessage} onDismiss={toggleToastActive} />
  ) : null;

  if (loading) {
    return (
      <Page title="Settings" subtitle="Configure your import and sync preferences">
        <Layout>
          <Layout.Section>
            <Card>
              <BlockStack gap="400">
                <SkeletonBodyText lines={3} />
                <SkeletonBodyText lines={2} />
              </BlockStack>
            </Card>
          </Layout.Section>
        </Layout>
      </Page>
    );
  }

  return (
  
    <>
      {toastMarkup}
      <Page title="Settings" subtitle="Configure your import and sync preferences">
        <Layout>
          <Layout.Section>
            <Banner tone="info">
              <Text>Sync features (Inventory Sync & Price Sync) are available in monthly plans only.</Text>
            </Banner>
          </Layout.Section>
          
          {error && (
            <Layout.Section>
              <Banner tone="critical">
                {error}
              </Banner>
            </Layout.Section>
          )}
          
          <Layout.Section>
            <Layout>
              <Layout.Section variant="oneHalf">
                <Card>
                  <BlockStack gap="400">
                    <Text variant="headingMd" as="h2">Product Import Settings</Text>
                    <FormLayout>
                      <Select
                        label="Products should be published?"
                        options={publishOptions}
                        value={publishStatus}
                        onChange={setPublishStatus}
                      />
                      <Select
                        label="Fulfillment service"
                        options={fulfillmentOptions}
                        value={fulfillmentService}
                        onChange={setFulfillmentService}
                      />
                      <Select
                        label="Inventory policy"
                        options={inventoryPolicyOptions}
                        value={inventoryPolicy}
                        onChange={setInventoryPolicy}
                      />
                    </FormLayout>
                  </BlockStack>
                </Card>
              </Layout.Section>
              
              <Layout.Section variant="oneHalf">
                <Card>
                  <BlockStack gap="400">
                    <Text variant="headingMd" as="h2">Sync Settings</Text>
                    <FormLayout>
                      <Select
                        label="Inventory Synchronization"
                        options={syncOptions}
                        value={inventorySync}
                        onChange={setInventorySync}
                        helpText="Sync inventory levels between Amazon and Shopify"
                      />
                      <Select
                        label="Price Synchronization"
                        options={syncOptions}
                        value={priceSync}
                        onChange={setPriceSync}
                        helpText="Sync prices between Amazon and Shopify"
                      />
                      <Select
                        label="Auto Import"
                        options={syncOptions}
                        value={autoImport}
                        onChange={setAutoImport}
                        helpText="Automatically import new products"
                      />
                    </FormLayout>
                  </BlockStack>
                </Card>
              </Layout.Section>
            </Layout>
          </Layout.Section>
          
          <Layout.Section>
            <InlineStack align="end">
              <Button 
                variant="primary"
                onClick={handleSave}
                loading={saving}
                disabled={saving}
              >
                {saving ? 'Saving...' : 'Save Settings'}
              </Button>
            </InlineStack>
          </Layout.Section>
        </Layout>
      </Page>
    </>
  );
}

export default Settings;