// Auth utility for token management
const AUTH_TOKEN_KEY = 'xpert_auth_token';
const USER_DATA_KEY = 'xpert_user_data';
const API_BASE_URL = import.meta.env.VITE_API_BASE_URL;
const API_KEY = import.meta.env.VITE_API_KEY;

export const authUtils = {
  // Store token and user data
  setAuthData(token, userData) {
    localStorage.setItem(AUTH_TOKEN_KEY, token);
    localStorage.setItem(USER_DATA_KEY, JSON.stringify(userData));
  },

  // Get stored token
  getToken() {
    return localStorage.getItem(AUTH_TOKEN_KEY);
  },

  // Get stored user data
  getUserData() {
    const userData = localStorage.getItem(USER_DATA_KEY);
    return userData ? JSON.parse(userData) : null;
  },

  // Clear auth data
  clearAuthData() {
    localStorage.removeItem(AUTH_TOKEN_KEY);
    localStorage.removeItem(USER_DATA_KEY);
  },

  // Check if token exists
  isAuthenticated() {
    return !!this.getToken();
  },

  // Re-authenticate when token expires
  async reAuthenticate() {
    try {
      const response = await fetch(`${API_BASE_URL}/authenticate?key=${API_KEY}`, {
        method: 'POST'
      });
      
      if (response.ok) {
        const data = await response.json();
        this.setAuthData(data.token, data.user);
        return data;
      } else {
        throw new Error('Re-authentication failed');
      }
    } catch (error) {
      console.error('Re-authentication error:', error);
      this.clearAuthData();
      throw error;
    }
  },

  // Make authenticated API request with auto re-auth
  async authenticatedRequest(url, options = {}) {
    let token = this.getToken();
    
    if (!token) {
      await this.reAuthenticate();
      token = this.getToken();
    }

    const requestOptions = {
      ...options,
      headers: {
        ...options.headers,
        'Authorization': `Bearer ${token}`,
        'Content-Type': 'application/json'
      }
    };

    try {
      const response = await fetch(url, requestOptions);
      
      // If token expired, re-authenticate and retry
      if (response.status === 401) {
        await this.reAuthenticate();
        const newToken = this.getToken();
        requestOptions.headers.Authorization = `Bearer ${newToken}`;
        return await fetch(url, requestOptions);
      }
      
      return response;
    } catch (error) {
      console.error('Authenticated request error:', error);
      throw error;
    }
  }
};